#!/usr/bin/env bash
set -euo pipefail

MIN_STEP=1
MAX_STEP=10

have_cmd() { command -v "$1" >/dev/null 2>&1; }

# Read integer percent (0..100)
get_percent() {
  if have_cmd brightnessctl; then
    # Try both CSV layouts
    local line f3 f5
    if line=$(brightnessctl -m get 2>/dev/null); then
      f3=$(awk -F, '{print $3}' <<<"$line")
      f5=$(awk -F, '{print $5}' <<<"$line")
      for f in "$f3" "$f5"; do
        if [[ "$f" =~ ^[0-9]+%$ ]]; then
          echo "${f%\%}"
          return 0
        fi
      done
      # Some versions output only raw values; fall through
    fi
  fi

  # Fallback via sysfs
  local dev cur max
  dev=$(ls -1 /sys/class/backlight 2>/dev/null | head -n1 || true)
  if [[ -n "$dev" && -r "/sys/class/backlight/$dev/brightness" && -r "/sys/class/backlight/$dev/max_brightness" ]]; then
    cur=$(<"/sys/class/backlight/$dev/brightness")
    max=$(<"/sys/class/backlight/$dev/max_brightness")
    if ((max > 0)); then
      echo $(((cur * 100 + max / 2) / max))
      return 0
    fi
  fi

  echo "Error: cannot determine current brightness percent." >&2
  exit 1
}

# Set absolute percent (1..100)
set_percent() {
  local pct="$1"
  if have_cmd brightnessctl; then
    brightnessctl set "${pct}%"
    return
  fi
  # sysfs fallback
  local dev max new_raw
  dev=$(ls -1 /sys/class/backlight 2>/dev/null | head -n1 || true)
  if [[ -z "$dev" ]]; then
    echo "Error: no /sys/class/backlight device found." >&2
    exit 1
  fi
  max=$(<"/sys/class/backlight/$dev/max_brightness")
  new_raw=$(((pct * max + 50) / 100))
  ((new_raw < 1)) && new_raw=1
  ((new_raw > max)) && new_raw=max
  echo "$new_raw" | sudo tee "/sys/class/backlight/$dev/brightness" >/dev/null
}

compute_step() {
  local pct="$1" step
  if ((pct <= 5)); then
    step=1
  elif ((pct <= 20)); then
    step=$(awk -v p="$pct" 'BEGIN{t=(p-5)/15; s=2 + t*(5-2); printf("%.0f", s)}')
  else
    step=$(awk -v p="$pct" 'BEGIN{t=(p-20)/80; s=5 + t*(10-5); printf("%.0f", s)}')
  fi
  ((step < MIN_STEP)) && step="$MIN_STEP"
  ((step > MAX_STEP)) && step="$MAX_STEP"
  echo "$step"
}

usage() {
  echo "Usage: $(basename "$0") {up|down}"
  exit 1
}

main() {
  [[ $# -eq 1 ]] || usage
  case "$1" in up | down) ;; *) usage ;; esac

  local before step target after
  before=$(get_percent)
  step=$(compute_step "$before")

  if [[ "$1" == "up" ]]; then
    target=$((before + step))
  else
    target=$((before - step))
  fi

  ((target < 1)) && target=1
  ((target > 100)) && target=100

  set_percent "$target"
  after=$(get_percent)
  echo "Brightness: ${before}% -> ${after}% (step ${step}%)"
}

main "$@"
